import Link from 'next/link';
import { usePathname } from 'next/navigation';
import {
  AlertCircle,
  Bell,
  Building2,
  CheckCircle,
  ChevronLeft,
  CreditCard,
  Crown,
  Globe,
  LogOut,
  Palette,
  Shield,
  Users,
  Zap,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';

const mockWorkspace = {
  id: '1',
  name: 'Acme Corporation',
  logo: '/media/brand-logos/acme.svg',
  plan: 'pro',
  status: 'active',
  members: 24,
  role: 'owner',
};

const workspaceMenuItems = [
  {
    id: 'overview',
    title: 'Workspace Overview',
    description: 'Manage your workspace settings and preferences',
    icon: Building2,
    href: '/crm/settings/workspace',
  },
  {
    id: 'members',
    title: 'Team Members',
    description: 'Invite, manage, and organize your team',
    icon: Users,
    href: '/crm/settings/members',
    badge: '24',
  },
  {
    id: 'billing',
    title: 'Billing & Plans',
    description: 'Manage subscriptions, invoices, and payment methods',
    icon: CreditCard,
    href: '/crm/settings/billing',
    badge: 'Pro Plan',
  },
  {
    id: 'integrations',
    title: 'Integrations',
    description: 'Connect with third-party tools and services',
    icon: Zap,
    href: '/crm/settings/integrations',
    badge: '8 Active',
  },
  {
    id: 'security',
    title: 'Security & Privacy',
    description: 'Manage authentication, permissions, and data protection',
    icon: Shield,
    href: '/crm/settings/security',
  },
  {
    id: 'appearance',
    title: 'Appearance',
    description: 'Customize themes, branding, and visual settings',
    icon: Palette,
    href: '/crm/settings/appearance',
  },
  {
    id: 'notifications',
    title: 'Notifications',
    description: 'Configure email, push, and in-app notifications',
    icon: Bell,
    href: '/crm/settings/notifications',
  },
  {
    id: 'regional',
    title: 'Regional Settings',
    description: 'Language, timezone, and localization preferences',
    icon: Globe,
    href: '/crm/settings/regional',
  },
];

export function SidebarWorkspace({ onSwitchToDefault }) {
  const pathname = usePathname();

  const getPlanBadgeVariant = (plan) => {
    switch (plan) {
      case 'enterprise':
        return 'destructive';
      case 'pro':
        return 'primary';
      default:
        return 'secondary';
    }
  };

  const getStatusIcon = (status) => {
    switch (status) {
      case 'active':
        return <CheckCircle className="size-3 text-emerald-500" />;
      case 'trial':
        return <AlertCircle className="size-3 text-amber-500" />;
      default:
        return <AlertCircle className="size-3 text-red-500" />;
    }
  };

  return (
    <>
      {/* Header - Same style as default sidebar */}
      <div className="group flex justify-between items-center gap-2.5 border-b border-border h-(--sidebar-header-height) shrink-0 px-2.5">
        <div className="flex items-center gap-2">
          <Button
            variant="ghost"
            size="sm"
            onClick={onSwitchToDefault}
            className="flex items-center gap-2 text-sm hover:bg-accent"
          >
            <ChevronLeft className="size-4" />
            <span className="in-data-[sidebar-collapsed]:hidden">
              Back to CRM
            </span>
          </Button>
        </div>
      </div>

      {/* Workspace Info - Compact version */}
      <div className="p-3 border-b border-border">
        <div className="flex items-center gap-2 mb-2">
          <Avatar className="size-8">
            <AvatarImage src={mockWorkspace.logo} alt={mockWorkspace.name} />
            <AvatarFallback className="text-sm font-semibold">
              {mockWorkspace.name
                .split(' ')
                .map((n) => n[0])
                .join('')}
            </AvatarFallback>
          </Avatar>
          <div className="flex-1 min-w-0">
            <h3 className="font-medium text-sm truncate">
              {mockWorkspace.name}
            </h3>
            <div className="flex items-center gap-1 mt-0.5">
              <Badge
                variant={getPlanBadgeVariant(mockWorkspace.plan)}
                size="sm"
              >
                {mockWorkspace.plan === 'enterprise' && (
                  <Crown className="size-3 mr-1" />
                )}
                {mockWorkspace.plan.charAt(0).toUpperCase() +
                  mockWorkspace.plan.slice(1)}
              </Badge>
              {getStatusIcon(mockWorkspace.status)}
            </div>
          </div>
        </div>
      </div>

      {/* Navigation Menu - Same style as default sidebar */}
      <nav className="flex-1 p-2 space-y-1">
        {workspaceMenuItems.map((item) => {
          const Icon = item.icon;
          const isActive = pathname === item.href;

          return (
            <Link
              key={item.id}
              href={item.href}
              className={cn(
                'flex items-center gap-3 px-3 py-2 rounded-lg text-sm transition-colors',
                'hover:bg-accent hover:text-accent-foreground',
                isActive
                  ? 'bg-accent text-accent-foreground'
                  : 'text-muted-foreground',
              )}
            >
              <Icon className="size-4" />
              <div className="flex-1 min-w-0">
                <div className="flex items-center justify-between">
                  <span className="font-medium truncate">{item.title}</span>
                  {item.badge && (
                    <Badge variant={item.variant || 'secondary'} size="sm">
                      {item.badge}
                    </Badge>
                  )}
                </div>
              </div>
            </Link>
          );
        })}
      </nav>

      {/* Footer - Same style as default sidebar */}
      <div className="flex items-center justify-between p-3 border-t border-border">
        <div className="flex items-center gap-2">
          <Avatar className="size-6">
            <AvatarImage src="/media/avatars/300-2.png" alt="User" />
            <AvatarFallback className="text-xs">JD</AvatarFallback>
          </Avatar>
          <span className="text-sm font-medium">John Doe</span>
        </div>
        <Button variant="ghost" size="sm">
          <LogOut className="size-3 mr-2" />
          Logs Out
        </Button>
      </div>
    </>
  );
}
